import {
  Button,
  Card,
  Col,
  Divider,
  Dropdown,
  Form,
  Icon,
  Input,
  Menu,
  Row,
  message,
  Popconfirm,
} from 'antd';
import React, { Component, Fragment } from 'react';
import { PageHeaderWrapper } from '@ant-design/pro-layout';
import { connect } from 'dva';
import StandardTable from './components/StandardTable';
import RoleCreateForm from './components/RoleCreateForm';
import RoleConfigForm from './components/RoleConfigForm';
import styles from './style.less';

const getValue = obj =>
  Object.keys(obj)
    .map(key => obj[key])
    .join(',');

/* eslint react/no-multi-comp:0 */
@connect(({ role, loading }) => ({
  role,
  loading: loading.models.role,
}))
class RoleManagement extends Component {
  /**
   * state
   * 维护组件唯一数据源
   * 允许存在以下数据源
   * Modal: 页面弹出框配置数据
   * ListOptions: 列表筛选器配置数据
   * List: 列表主要数据源
   * Selected: 选中相关的
   */
  state = {
    roleCreateModalVisible: false,
    roleConfigModalVisible: false,
    selectedRows: [],
    listOptions: {},
    currentRecord: {},
    pagination: {
      pageNum: 1,
      pageSize: 10,
    },
  };

  columns = [
    {
      title: '角色名称',
      dataIndex: 'roleName',
    },
    {
      title: '操作',
      render: (text, record) => (
        <Fragment>
          <a onClick={() => this.handleOpenRoleConfigModal('view', record)}>查看</a>
          <Divider type="vertical" />
          <a onClick={() => this.handleOpenRoleConfigModal('edit', record)}>编辑</a>
          <Divider type="vertical" />
          <Popconfirm
            title="您确定要移除该角色吗?"
            onConfirm={() => this.handleDeleteRoleRecord(true, record)}
            okText="确认"
            cancelText="取消"
          >
            <a>删除</a>
          </Popconfirm>
        </Fragment>
      ),
    },
  ];

  componentDidMount() {
    const { pagination } = this.state;
    this.getRoles({
      ...pagination,
    });
  }

  /**
   * private methods
   */

  getRoles = payload => {
    const { dispatch } = this.props;
    dispatch({
      type: 'role/getRoles',
      payload: {
        ...payload,
      },
    });
  };

  refreshList = () => {
    const { pagination, listOptions } = this.state;
    this.getRoles({
      ...pagination,
      ...listOptions,
    });
  };

  /**
   * handlers
   */
  handleStandardTableChange = (pagination, filtersArg, sorter) => {
    const { dispatch } = this.props;
    const { listOptions } = this.state;
    const filters = Object.keys(filtersArg).reduce((obj, key) => {
      const newObj = { ...obj };
      newObj[key] = getValue(filtersArg[key]);
      return newObj;
    }, {});
    this.setState({
      pagination: {
        pageNum: pagination.current,
        pageSize: pagination.pageSize,
      },
    });
    const params = {
      ...pagination,
      ...listOptions,
      ...filters,
    };

    if (sorter.field) {
      params.sorter = `${sorter.field}_${sorter.order}`;
    }

    dispatch({
      type: 'role/getRoles',
      payload: params,
    });
  };

  handleFormReset = () => {
    const { form } = this.props;
    const { pagination } = this.state;
    form.resetFields();
    this.setState({
      listOptions: {},
    });
    this.getRoles({
      ...pagination,
    });
  };

  handleBatchOperationMenuClick = e => {
    const { dispatch } = this.props;
    const { selectedRows } = this.state;
    if (!selectedRows) return;

    switch (e.key) {
      case 'remove':
        dispatch({
          type: 'listAndtableAndlist/remove',
          payload: {
            key: selectedRows.map(row => row.key),
          },
          callback: () => {
            this.setState({
              selectedRows: [],
            });
          },
        });
        break;

      default:
        break;
    }
  };

  handleSelectRows = rows => {
    this.setState({
      selectedRows: rows,
    });
  };

  handleSearch = e => {
    e.preventDefault();
    const { dispatch, form } = this.props;
    const { pagination } = this.state;
    form.validateFields((err, fieldsValue) => {
      if (err) return;
      const values = {
        ...fieldsValue,
      };
      this.setState({
        listOptions: values,
        pagination: {
          pageNum: 1,
          pageSize: 10,
        },
      });
      dispatch({
        type: 'role/getRoles',
        payload: {
          ...values,
          ...pagination,
        },
      });
    });
  };

  handleRoleCreateModalVisible = flag => {
    this.setState({
      roleCreateModalVisible: !!flag,
    });
  };

  handleRoleConfigModalVisible = flag => {
    this.setState({
      roleConfigModalVisible: !!flag,
    });
  };

  handleAddRole = fields => {
    const { dispatch } = this.props;
    dispatch({
      type: 'role/addRole',
      payload: {
        ...fields,
      },
      callback: () => {
        message.success('操作成功');
        this.refreshList();
        this.handleRoleCreateModalVisible();
      },
    });
  };

  handleOpenRoleConfigModal = (type, record) => {
    const { dispatch } = this.props;
    dispatch({
      type: 'role/getRoleInfo',
      payload: { ...record },
    });
    this.setState({
      currentRecord: {
        ...record,
        actionType: type,
      },
    });
    this.handleRoleConfigModalVisible(true);
  };

  handleUpdateRole = values => {
    const { dispatch } = this.props;
    dispatch({
      type: 'role/updateRoleInfo',
      payload: { ...values },
      callback: () => {
        message.success('操作成功');
      },
    });
  };

  handleDeleteRoleRecord = (flag, record) => {
    if (!flag) {
      return;
    }
    const { dispatch } = this.props;
    dispatch({
      type: 'role/deleteRole',
      payload: {
        id: record.id,
      },
      callback: () => {
        message.success('操作成功');
        this.refreshList();
      },
    });
  };

  /**
   * renders
   */

  renderFilter() {
    // you can use expandForm or another key to expand and fold an complex form
    const { form } = this.props;
    const { getFieldDecorator } = form;
    return (
      <Form onSubmit={this.handleSearch} layout="inline">
        <Row
          gutter={{
            md: 8,
            lg: 24,
            xl: 48,
          }}
        >
          <Col md={8} sm={24}>
            <Form.Item label="角色名称">
              {getFieldDecorator('roleName')(<Input placeholder="请输入" />)}
            </Form.Item>
          </Col>
          <Col md={8} sm={24}>
            <span className={styles.submitButtons}>
              <Button type="primary" htmlType="submit">
                查询
              </Button>
              <Button
                style={{
                  marginLeft: 8,
                }}
                onClick={this.handleFormReset}
              >
                重置
              </Button>
            </span>
          </Col>
        </Row>
      </Form>
    );
  }

  render() {
    const {
      role: { users },
      loading,
    } = this.props;

    const {
      selectedRows,
      roleCreateModalVisible,
      roleConfigModalVisible,
      currentRecord,
    } = this.state;
    const batchOperationMenu = (
      <Menu onClick={this.handleBatchOperationMenuClick} selectedKeys={[]}>
        <Menu.Item key="remove">删除</Menu.Item>
      </Menu>
    );
    const roleCreateModalMethods = {
      handleAddRole: this.handleAddRole,
      handleModalVisible: this.handleRoleCreateModalVisible,
    };
    const roleConfigModalMethods = {
      handleModalVisible: this.handleRoleConfigModalVisible,
      handleUpdateRole: this.handleUpdateRole,
    };
    return (
      <PageHeaderWrapper>
        <Card bordered={false}>
          <div className={styles.tableList}>
            <div className={styles.tableListForm}>{this.renderFilter()}</div>
            <div className={styles.tableListOperator}>
              <Button
                icon="plus"
                type="primary"
                onClick={() => this.handleRoleCreateModalVisible(true)}
              >
                新建
              </Button>
              {selectedRows.length > 0 && (
                <span>
                  <Dropdown overlay={batchOperationMenu}>
                    <Button>
                      批量操作 <Icon type="down" />
                    </Button>
                  </Dropdown>
                </span>
              )}
            </div>
            <StandardTable
              selectedRows={selectedRows}
              loading={loading}
              data={users}
              columns={this.columns}
              onSelectRow={this.handleSelectRows}
              onChange={this.handleStandardTableChange}
            />
          </div>
        </Card>
        <RoleCreateForm {...roleCreateModalMethods} modalVisible={roleCreateModalVisible} />
        <RoleConfigForm
          {...roleConfigModalMethods}
          record={currentRecord}
          modalVisible={roleConfigModalVisible}
        />
      </PageHeaderWrapper>
    );
  }
}

export default Form.create()(RoleManagement);
